
/*
 * The olsr.org Optimized Link-State Routing daemon version 2 (olsrd2)
 * Copyright (c) 2004-2015, the olsr.org team - see HISTORY file
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 * * Neither the name of olsr.org, olsrd nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Visit http://www.olsr.org for more information.
 *
 * If you find this software useful feel free to make a donation
 * to the project. For more information see the website or contact
 * the copyright holders.
 *
 */

#ifndef OS_ROUTING_H_
#define OS_ROUTING_H_

#include <stdio.h>
#include <sys/time.h>

#include "common/common_types.h"
#include "common/list.h"
#include "common/netaddr.h"
#include "subsystems/oonf_interface.h"
#include "core/oonf_logging.h"
#include "subsystems/os_system.h"

#define OONF_OS_ROUTING_SUBSYSTEM "os_routing"

/* include os-specific headers */
#if defined(__linux__)
#include "subsystems/os_linux/os_routing_linux.h"
#elif defined (BSD)
#include "subsystems/os_bsd/os_routing_bsd.h"
#elif defined (_WIN32)
#include "subsystems/os_win32/os_routing_win32.h"
#else
#error "Unknown operation system"
#endif

/* make sure default values for routing are there */
#ifndef RTPROT_UNSPEC
#define RTPROT_UNSPEC 0
#endif
#ifndef RT_TABLE_UNSPEC
#define RT_TABLE_UNSPEC 0
#endif

struct os_route_str {
  char buf[
           /* header */
           1+
           /* src-ip */
           8 + sizeof(struct netaddr_str)
           /* gw */
           + 4 + sizeof(struct netaddr_str)
           /* dst [type] */
           + 5 + sizeof(struct netaddr_str)
           /* src-prefix */
           + 12 + sizeof(struct netaddr_str)
           /* metric */
           + 7+11
           /* table, protocol */
           +6+4 +9+4
           +3 + IF_NAMESIZE + 2 + 10 + 2
           /* footer and 0-byte */
           + 2];
};

enum os_route_type {
  OS_ROUTE_UNDEFINED,
  OS_ROUTE_UNICAST,
  OS_ROUTE_LOCAL,
  OS_ROUTE_BROADCAST,
  OS_ROUTE_MULTICAST,
  OS_ROUTE_THROW,
  OS_ROUTE_UNREACHABLE,
  OS_ROUTE_PROHIBIT,
  OS_ROUTE_BLACKHOLE,
  OS_ROUTE_NAT,
};

struct os_route_key {
  struct netaddr dst;
  struct netaddr src;
};

struct os_route {
  /* used for delivering feedback about netlink commands */
  struct os_route_internal _internal;

  /* address family */
  unsigned char family;

  /* type of route */
  enum os_route_type type;

  /* combination of source and destination */
  struct os_route_key key;

  /* gateway and destination */
  struct netaddr gw;

  /* source IP that should be used for outgoing IP packets of this route */
  struct netaddr src_ip;

  /* metric of the route */
  int metric;

  /* routing table and routing protocol */
  unsigned char table, protocol;

  /* index of outgoing interface */
  unsigned int if_index;

  /* callback when operation is finished */
  void (*cb_finished)(struct os_route *, int error);

  /* callback for os_routing_query() */
  void (*cb_get)(struct os_route *filter, struct os_route *route);
};

struct os_route_listener {
  /* used for delivering feedback about netlink commands */
  struct os_route_listener_internal _internal;

  /* callback called for each routing event */
  void (*cb_get)(const struct os_route *route, bool set);
};

/* prototypes for all os_routing functions */
EXPORT bool os_routing_supports_source_specific(int af_family);
EXPORT int os_routing_set(struct os_route *, bool set, bool del_similar);
EXPORT int os_routing_query(struct os_route *);
EXPORT void os_routing_interrupt(struct os_route *);
EXPORT bool os_routing_is_in_progress(struct os_route *);

EXPORT void os_routing_listener_add(struct os_route_listener *);
EXPORT void os_routing_listener_remove(struct os_route_listener *);

EXPORT const char *os_routing_to_string(
    struct os_route_str *buf, const struct os_route *route);

EXPORT const struct os_route *os_routing_get_wildcard_route(void);

EXPORT int os_route_avl_cmp_route_key(const void *, const void *);
EXPORT void os_route_init_half_os_route_key(
    struct netaddr *any, struct netaddr *specific,
    const struct netaddr *source);

static INLINE void
os_route_init_sourcespec_prefix(struct os_route_key *prefix,
    const struct netaddr *destination) {
  os_route_init_half_os_route_key(&prefix->src, &prefix->dst, destination);
}

static INLINE void
os_route_init_sourcespec_src_prefix(struct os_route_key *prefix,
    const struct netaddr *source) {
  os_route_init_half_os_route_key(&prefix->dst, &prefix->src, source);
}


#endif /* OS_ROUTING_H_ */
